<?php

namespace TourGuide\Widgets;

use TourGuide\TourGuideUtils;

class Widget
{
    public $id;
    public $widget_id;
    public $title;
    public $description;
    public $type;
    public $status;
    public $selected_tours = [];
    public $settings = [];
    public $domain_whitelist = [];
    public $created_at;
    public $updated_at;

    /**
     * Constructor for the Widget model. Initializes the widget properties from the provided data array.
     * 
     * @param array $data Associative array containing widget data.
     */
    public function __construct(array $data = [])
    {
        if (!empty($data)) {
            $this->id = $data['id'] ?? null;
            $this->widget_id = $data['widget_id'] ?? '';
            $this->title = $data['title'] ?? '';
            $this->description = $data['description'] ?? '';
            $this->type = $data['type'] ?? 'player';
            $this->status = $data['status'] ?? 'active';
            $this->created_at = $data['created_at'] ?? '';
            $this->updated_at = $data['updated_at'] ?? '';

            // Decode and clean widget data
            $this->selected_tours = $this->decodeJsonField($data['selected_tours'] ?? '');
            $this->settings = $this->decodeJsonField($data['settings'] ?? '');
            $this->domain_whitelist = $this->decodeJsonField($data['domain_whitelist'] ?? '');
        }
    }

    /**
     * Generate a unique widget ID.
     *
     * @return string Unique widget identifier.
     */
    public static function generateWidgetId()
    {
        return 'tg_' . uniqid() . '_' . substr(md5(time() . rand()), 0, 8);
    }

    /**
     * Get the embed code for this widget.
     *
     * @param string $baseUrl The base URL for the tour guide system.
     * @return string The HTML embed code.
     */
    public function getEmbedCode($baseUrl = '')
    {
        $scriptUrl = rtrim($baseUrl, '/') . '/widget/' . $this->widget_id . '.js';
        
        return sprintf(
            '<script src="%s" data-widget-id="%s" data-widget-type="%s"></script>',
            TourGuideUtils::escapeHtml($scriptUrl),
            TourGuideUtils::escapeHtml($this->widget_id),
            TourGuideUtils::escapeHtml($this->type)
        );
    }

    /**
     * Get widget configuration for JavaScript rendering.
     *
     * @return array Widget configuration array.
     */
    public function getWidgetConfig()
    {
        return [
            'widget_id' => $this->widget_id,
            'type' => $this->type,
            'title' => $this->title,
            'selected_tours' => $this->selected_tours,
            'settings' => array_merge($this->getDefaultSettings(), $this->settings),
            'domain_whitelist' => $this->domain_whitelist,
        ];
    }

    /**
     * Get default widget settings.
     *
     * @return array Default settings.
     */
    public function getDefaultSettings()
    {
        return [
            'theme' => 'default',
            'position' => 'bottom-right',
            'auto_start' => false,
            'show_progress' => true,
            'allow_skip' => true,
            'button_text' => tourGuideHelper()->translate('tour_guide_start_tour'),
            'completion_message' => tourGuideHelper()->translate('tour_guide_tour_completed'),
        ];
    }

    /**
     * Converts the Widget instance to an associative array format.
     *
     * @return array The widget represented as an associative array.
     */
    public function toArray()
    {
        return [
            'id' => $this->id,
            'widget_id' => $this->widget_id,
            'title' => $this->title,
            'description' => $this->description,
            'type' => $this->type,
            'status' => $this->status,
            'selected_tours' => $this->selected_tours,
            'settings' => $this->settings,
            'domain_whitelist' => $this->domain_whitelist,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
        ];
    }

    /**
     * Converts the Widget instance to a JSON string format.
     *
     * @return string The widget represented as a JSON string.
     */
    public function toJson()
    {
        return json_encode($this->toArray(), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }

    /**
     * Decodes a JSON-encoded string, returning an array or defaulting to an empty array if decoding fails.
     * 
     * @param mixed $field JSON string to decode.
     * @return array Decoded JSON data.
     */
    public static function decodeJsonField($field)
    {
        if (is_array($field) || is_object($field)) return (array)$field;
        
        if (empty($field)) return [];

        return json_decode($field, true) ?? [];
    }

    /**
     * Encodes data as JSON for database storage.
     * 
     * @param mixed $data Data to encode.
     * @return string JSON encoded string.
     */
    public static function encodeJsonField($data)
    {
        return json_encode($data, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Prepare data array for saving to the database.
     *
     * @param array $data The data to prepare for saving.
     * @return array Array data ready for saving to the database.
     */
    public static function prepareForSaving(array $data): array
    {
        // Define the keys that need JSON encoding
        $keysToEncode = ['selected_tours', 'settings', 'domain_whitelist'];

        foreach ($keysToEncode as $key) {
            if (isset($data[$key]) && !is_string($data[$key])) {
                $data[$key] = self::encodeJsonField($data[$key]);
            }
        }

        // Generate widget_id if not provided
        if (empty($data['widget_id'])) {
            $data['widget_id'] = self::generateWidgetId();
        }

        return $data;
    }

    /**
     * Validate widget data.
     *
     * @param array $data Widget data to validate.
     * @return array Array of validation errors (empty if valid).
     */
    public static function validate(array $data): array
    {
        $errors = [];

        if (empty($data['title'])) {
            $errors[] = tourGuideHelper()->translate('tour_guide_widget_title_required');
        }

        if (!in_array($data['type'] ?? '', ['setup', 'player'])) {
            $errors[] = tourGuideHelper()->translate('tour_guide_widget_type_invalid');
        }

        if (!in_array($data['status'] ?? '', ['active', 'inactive'])) {
            $errors[] = tourGuideHelper()->translate('tour_guide_widget_status_invalid');
        }

        return $errors;
    }
}