<?php

namespace TourGuide\Models;

use TourGuide\TourGuideUtils;

class Tour
{
    public $id;
    public $title;
    public $description;
    public $priority;
    public $status;
    public $steps = [];
    public $settings = [];
    public $steps_translations = [];
    public $created_at;

    /**
     * Constructor for the Tour model. Initializes the tour properties from the provided data array.
     * 
     * @param array $data Associative array containing tour data.
     */
    public function __construct(array $data = [])
    {
        if (!empty($data)) {
            $this->id = $data['id'] ?? null;
            $this->title = $data['title'] ?? '';
            $this->description = $data['description'] ?? '';
            $this->priority = $data['priority'] ?? 0;
            $this->status = $data['status'] ?? 'inactive';
            $this->created_at = $data['created_at'] ?? '';

            // Decode and clean tour steps
            $this->steps = $this->decodeSteps($data['steps'] ?? '');

            // Decode and parse settings
            $this->settings = $this->decodeSettings($data['settings'] ?? '');

            // Decode and clean steps translations
            $this->steps_translations = $this->decodeTranslations($data['steps_translations'] ?? []);
        }
    }

    /**
     * Decodes the base64-encoded JSON steps data, processes each step to decode any 
     * HTML entities and slashes, and decodes click track data within each step.
     * 
     * @param string $steps Encoded JSON string of steps.
     * @return array Decoded and cleaned steps data.
     */
    public function decodeSteps($steps)
    {
        $decodedSteps = $this->decodeJsonField($steps);

        foreach ($decodedSteps as $key => &$step) {
            // Clean and decode key properties of each step
            $step['title'] = $this->cleanString($step['title'] ?? '');
            $step['description'] = $this->cleanString($step['description'] ?? '');
            $step['selector'] = $this->cleanString($step['selector'] ?? '');

            // If click tracks exist, decode their selectors
            if (isset($step['clickTracks'])) {
                $step['clickTracks'] = $this->decodeClickTracks($step['clickTracks']);
            }
        }

        return $decodedSteps;
    }

    /**
     * Decodes and cleans each selector within the clickTracks data for a tour step.
     * 
     * @param array $clickTracks Array of click track data to decode.
     * @return array Cleaned click track data with decoded selectors.
     */
    public function decodeClickTracks(array $clickTracks)
    {
        foreach ($clickTracks as &$track) {
            foreach ($track as &$click) {
                $click['selector'] = $this->cleanString($click['selector'] ?? '');
            }
        }
        return $clickTracks;
    }

    /**
     * Decodes the steps translations for multiple languages and cleans both title and description.
     * 
     * @param string $translations JSON-encoded translations data.
     * @return array Decoded and cleaned translations.
     */
    public function decodeTranslations($translations)
    {
        $translations = $this->decodeJsonField($translations);

        foreach ($translations as $language => &$steps) {
            foreach ($steps as &$step) {
                $step['title'] = $this->cleanString($step['title'] ?? '');
                $step['description'] = $this->cleanString($step['description'] ?? '');
            }
        }

        return $translations;
    }

    /**
     * Decodes the tour settings including triggers
     * 
     * @param string $settings JSON-encoded settings data.
     * @return array Decoded settings.
     */
    public function decodeSettings($settings)
    {
        $decodedSettings = $this->decodeJsonField($settings);

        if (isset($decodedSettings['triggers'])) {
            $decodedSettings['triggers'] = $this->decodeJsonField($decodedSettings['triggers']);
        }

        return $decodedSettings;
    }


    /**
     * Converts the Tour instance to an associative array format.
     *
     * @return array The tour represented as an associative array.
     */
    public function toArray()
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            'description' => $this->description,
            'priority' => $this->priority,
            'status' => $this->status,
            'steps' => $this->steps,
            'settings' => $this->settings,
            'steps_translations' => $this->steps_translations,
            'created_at' => $this->created_at,
        ];
    }

    /**
     * Converts the Tour instance to a JSON string format.
     *
     * @return string The tour represented as a JSON string.
     */
    public function toJson()
    {
        return json_encode($this->toArray(), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }

    /**
     * Decodes a JSON-encoded string, returning an array or defaulting to an empty array if decoding fails.
     * 
     * @param array $field JSON string to decode.
     * @return array Decoded JSON data.
     */
    public static function encodeJsonField($data)
    {
        // JSON encode the key's value
        $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

        // Check for JSON encoding errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception(json_last_error_msg(), json_last_error());
        }

        // Base64 encode the JSON string for safe database storage
        return base64_encode($json);
    }


    /**
     * Decodes a JSON-encoded string, returning an array or defaulting to an empty array if decoding fails.
     * 
     * @param mixed $field JSON string to decode.
     * @return array Decoded JSON data.
     */
    public static function decodeJsonField($field)
    {
        if (is_array($field) || is_object($field)) return (array)$field;

        $fieldDecoded = base64_decode($field, true);
        if ($fieldDecoded === false)
            $fieldDecoded = $field;

        return json_decode($fieldDecoded, true) ?? [];
    }

    /**
     * Cleans a given string by decoding HTML entities and stripping slashes.
     * This function is intended for use with user-generated input that may have been 
     * sanitized or escaped on save, ensuring consistent formatting.
     * 
     * @param string $string The string to clean.
     * @return string Cleaned string after applying html_entity_decode and stripslashes.
     */
    public static function cleanString($string)
    {
        $string = TourGuideUtils::sanitizeData($string);
        return stripslashes(html_entity_decode($string));
    }

    /**
     * Prepare data array for saving to the database.
     *
     * This method encodes and formats specified keys (like `steps`, `settings`, `steps_translations`)
     * as JSON strings, then base64 encodes them for consistent storage and database safety. It also allows for 
     * handling subkeys dynamically, where subkeys may have special processing requirements.
     *
     * @param array $data The data to prepare for saving.
     * @param array $subkeyHandlers Optional list of handlers for special subkey processing.
     * @return array Array data ready for saving to the database.
     * @throws \Exception if JSON encoding fails.
     */
    public static function prepareForSaving(array $data, array $subkeyHandlers = []): array
    {
        // Define the keys that need JSON and base64 encoding
        $keysToEncode = ['steps', 'settings', 'steps_translations'];

        // Common subkey handler for JSON decoding with fallback to stripslashes
        $jsonStringCommonHandler = function ($value) {

            if (is_array($value)) return $value;
            if (empty($value)) return [];

            // Ensure value is decoded correctly
            $decoded = json_decode($value, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                // Try again with stripslashes if decoding fails
                $decoded = json_decode(stripslashes($value), true);
            }
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new \Exception("Invalid JSON: " . json_last_error_msg());
            }
            return $decoded;
        };

        // Ensure the 'settings' key exists and add the 'triggers' subkey handler if needed
        if (!isset($subkeyHandlers['settings']['triggers'])) {
            if (!isset($subkeyHandlers['settings'])) {
                $subkeyHandlers['settings'] = [];
            }
            $subkeyHandlers['settings']['triggers'] = $jsonStringCommonHandler;
        }

        foreach ($keysToEncode as $key) {
            if (isset($data[$key])) {
                // Check if there are specific handlers for subkeys of this key
                if (isset($subkeyHandlers[$key])) {
                    foreach ($subkeyHandlers[$key] as $subkey => $handler) {
                        if (isset($data[$key][$subkey])) {
                            try {
                                // Apply the handler for the subkey if available
                                $data[$key][$subkey] = $handler($data[$key][$subkey]);
                            } catch (\Throwable $th) {
                                throw new \Exception("Subkey Handler Error for key '$key.$subkey': " . $th->getMessage(), $th->getCode());
                            }
                        }
                    }
                }

                try {

                    // Base64 encode the JSON string for safe database storage
                    $data[$key] = self::encodeJsonField($data[$key]);
                } catch (\Throwable $th) {
                    throw new \Exception("JSON encoding error for key '$key': " . $th->getMessage(), $th->getCode());
                }
            }
        }

        return $data;
    }
}